﻿#
# This script handles common telemetry tasks for Install.ps1 and Add-AppDevPackage.ps1.
#

function IsVsTelemetryRegOptOutSet()
{
    $VsTelemetryRegOptOutKeys = @(
        "HKLM:\SOFTWARE\Policies\Microsoft\VisualStudio\SQM",
        "HKLM:\SOFTWARE\Wow6432Node\Microsoft\VSCommon\16.0\SQM",
        "HKLM:\SOFTWARE\Microsoft\VSCommon\16.0\SQM",
        "HKLM:\SOFTWARE\Wow6432Node\Microsoft\VSCommon\17.0\SQM",
        "HKLM:\SOFTWARE\Microsoft\VSCommon\17.0\SQM"
    )

    foreach ($optOutKey in $VsTelemetryRegOptOutKeys)
    {
        if (Test-Path $optOutKey)
        {
            # If any of these keys have the DWORD value OptIn set to 0 that means that the user
            # has explicitly opted out of logging telemetry from Visual Studio.
            $val = (Get-ItemProperty $optOutKey)."OptIn"
            if ($val -eq 0)
            {
                return $true
            }
        }
    }

    return $false
}

try
{
    $TelemetryOptedOut = IsVsTelemetryRegOptOutSet
    if (!$TelemetryOptedOut)
    {
        $TelemetryAssembliesFolder = $args[0]
        $EventName = $args[1]
        $ReturnCode = $args[2]
        $ProcessorArchitecture = $args[3]

        foreach ($file in Get-ChildItem $TelemetryAssembliesFolder -Filter "*.dll")
        {
            [Reflection.Assembly]::LoadFile((Join-Path $TelemetryAssembliesFolder $file)) | Out-Null
        }

        [Microsoft.VisualStudio.Telemetry.TelemetryService]::DefaultSession.IsOptedIn = $True
        [Microsoft.VisualStudio.Telemetry.TelemetryService]::DefaultSession.Start()

        $TelEvent = New-Object "Microsoft.VisualStudio.Telemetry.TelemetryEvent" -ArgumentList $EventName
        if ($null -ne $ReturnCode)
        {
            $TelEvent.Properties["VS.DesignTools.SideLoadingScript.ReturnCode"] = $ReturnCode
        }

        if ($null -ne $ProcessorArchitecture)
        {
            $TelEvent.Properties["VS.DesignTools.SideLoadingScript.ProcessorArchitecture"] = $ProcessorArchitecture
        }

        [Microsoft.VisualStudio.Telemetry.TelemetryService]::DefaultSession.PostEvent($TelEvent)
        [Microsoft.VisualStudio.Telemetry.TelemetryService]::DefaultSession.Dispose()
    }
}
catch
{
    # Ignore telemetry errors
}
# SIG # Begin signature block
# MIImQwYJKoZIhvcNAQcCoIImNDCCJjACAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBs5RUy/PKaA/U5
# /+cdRqsBJInUnLtOhcpeK3MamAG/SqCCC3YwggT+MIID5qADAgECAhMzAAAFVskg
# Kx90Ml0tAAAAAAVWMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTAwHhcNMjMxMDE5MTk1MTExWhcNMjQxMDE2MTk1MTExWjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQCW2kg89zWnssiDG9AFq/sxqkPT43V4MZ6vwu7C4+sly46QJA4UNxSRnGDz9u8f
# 18OR1/Ar9guQDqRthrOOktLaol49HZfv8l1DEeB/uguWS0qFIJe+cK6b7grVfl0S
# Z7U8yOXu67n52z+QJN+iUxcIk0KWcu/sJZ8KfewKxnNDMDIHHhUHvIjQWhvKxOLC
# /TwL0i4wTykXuQwbwIZuuGi7TYA82Q2LkYOLlwOTPOb0eowhcUI7CaXlU30IgoG4
# frY+mtZMZnTx0ga/a36GVNdq6NeBKzlGbLB02/WIP+c5ADDCXsheWpkizJt9aPhz
# Yuont+Nx0OhIH04lJFvCqjljAgMBAAGjggF9MIIBeTAfBgNVHSUEGDAWBgorBgEE
# AYI3PQYBBggrBgEFBQcDAzAdBgNVHQ4EFgQUJt8FhQFwX7g5A/0elDxkjww82dow
# VAYDVR0RBE0wS6RJMEcxLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVsYW5kIE9wZXJh
# dGlvbnMgTGltaXRlZDEWMBQGA1UEBRMNMjMwODY1KzUwMTY1NTAfBgNVHSMEGDAW
# gBTm/F97uyIAWORyTrX0IXQjMubvrDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8v
# Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
# MjAxMC0wNy0wNi5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRw
# Oi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8yMDEw
# LTA3LTA2LmNydDAMBgNVHRMBAf8EAjAAMA0GCSqGSIb3DQEBCwUAA4IBAQBCnZlo
# MxjHBVHJEP6d+0t7EDJfNB0h7j9iOX6LQiPMdQAV1xqA6Jgb/VEPSEVs076CAJn3
# KP5Jpr4DOuo12BE2CoCusZDHn+7aP2yxngMGGQcJKz4UfOzDMS9Yr2sl3R6fMP18
# T4Ce5t7IYFK36PI/HfvHO8AzB3T0KhVIifqsSwZ8mEukw/wi4duJYulYxV2lY2kM
# Hwv/OKvKAsrTd338HlnuR+z9NZvStNUi1a8sTwcEMrEP8lgtoxaZe0Ypu/Ohn4Pq
# TefEVWKQJLU2KFyLDkoamDde0ydAzzHYEqdBQMW0gxWyvJ9fjgF/C2PKQVu/0SMB
# hcwo04eFzf/w7pZ/MIIGcDCCBFigAwIBAgIKYQxSTAAAAAAAAzANBgkqhkiG9w0B
# AQsFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
# BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAG
# A1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAw
# HhcNMTAwNzA2MjA0MDE3WhcNMjUwNzA2MjA1MDE3WjB+MQswCQYDVQQGEwJVUzET
# MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
# TWljcm9zb2Z0IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBT
# aWduaW5nIFBDQSAyMDEwMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
# 6Q5kUHlntcTj/QkATJ6UrPdWaOpE2M/FWE+ppXZ8bUW60zmStKQe+fllguQX0o/9
# RJwI6GWTzixVhL99COMuK6hBKxi3oktuSUxrFQfe0dLCiR5xlM21f0u0rwjYzIjW
# axeUOpPOJj/s5v40mFfVHV1J9rIqLtWFu1k/+JC0K4N0yiuzO0bj8EZJwRdmVMkc
# vR3EVWJXcvhnuSUgNN5dpqWVXqsogM3Vsp7lA7Vj07IUyMHIiiYKWX8H7P8O7YAS
# NUwSpr5SW/Wm2uCLC0h31oVH1RC5xuiq7otqLQVcYMa0KlucIxxfReMaFB5vN8sZ
# M4BqiU2jamZjeJPVMM+VHwIDAQABo4IB4zCCAd8wEAYJKwYBBAGCNxUBBAMCAQAw
# HQYDVR0OBBYEFOb8X3u7IgBY5HJOtfQhdCMy5u+sMBkGCSsGAQQBgjcUAgQMHgoA
# UwB1AGIAQwBBMAsGA1UdDwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQY
# MBaAFNX2VsuP6KJcYmjRPZSQW9fOmhjEMFYGA1UdHwRPME0wS6BJoEeGRWh0dHA6
# Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1
# dF8yMDEwLTA2LTIzLmNybDBaBggrBgEFBQcBAQROMEwwSgYIKwYBBQUHMAKGPmh0
# dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2VydHMvTWljUm9vQ2VyQXV0XzIw
# MTAtMDYtMjMuY3J0MIGdBgNVHSAEgZUwgZIwgY8GCSsGAQQBgjcuAzCBgTA9Bggr
# BgEFBQcCARYxaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL1BLSS9kb2NzL0NQUy9k
# ZWZhdWx0Lmh0bTBABggrBgEFBQcCAjA0HjIgHQBMAGUAZwBhAGwAXwBQAG8AbABp
# AGMAeQBfAFMAdABhAHQAZQBtAGUAbgB0AC4gHTANBgkqhkiG9w0BAQsFAAOCAgEA
# GnTvV08pe8QWhXi4UNMi/AmdrIKX+DT/KiyXlRLl5L/Pv5PI4zSp24G43B4AvtI1
# b6/lf3mVd+UC1PHr2M1OHhthosJaIxrwjKhiUUVnCOM/PB6T+DCFF8g5QKbXDrMh
# KeWloWmMIpPMdJjnoUdD8lOswA8waX/+0iUgbW9h098H1dlyACxphnY9UdumOUjJ
# N2FtB91TGcun1mHCv+KDqw/ga5uV1n0oUbCJSlGkmmzItx9KGg5pqdfcwX7RSXCq
# tq27ckdjF/qm1qKmhuyoEESbY7ayaYkGx0aGehg/6MUdIdV7+QIjLcVBy78dTMgW
# 77Gcf/wiS0mKbhXjpn92W9FTeZGFndXS2z1zNfM8rlSyUkdqwKoTldKOEdqZZ14y
# jPs3hdHcdYWch8ZaV4XCv90Nj4ybLeu07s8n07VeafqkFgQBpyRnc89NT7beBVaX
# evfpUk30dwVPhcbYC/GO7UIJ0Q124yNWeCImNr7KsYxuqh3khdpHM2KPpMmRM19x
# HkCvmGXJIuhCISWKHC1g2TeJQYkqFg/XYTyUaGBS79ZHmaCAQO4VgXc+nOBTGBpQ
# HTiVmx5mMxMnORd4hzbOTsNfsvU9R1O24OXbC2E9KteSLM43Wj5AQjGkHxAIwlac
# vyRdUQKdannSF9PawZSOB3slcUSrBmrm1MbfI5qWdcUxghojMIIaHwIBATCBlTB+
# MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
# bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9N
# aWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQSAyMDEwAhMzAAAFVskgKx90Ml0tAAAA
# AAVWMA0GCWCGSAFlAwQCAQUAoIGuMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEE
# MBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCAN
# SiN/A9m3PfXxWvXOr5Y6vf4QWtWaOKTWzL+fiYGStDBCBgorBgEEAYI3AgEMMTQw
# MqAUgBIATQBpAGMAcgBvAHMAbwBmAHShGoAYaHR0cDovL3d3dy5taWNyb3NvZnQu
# Y29tMA0GCSqGSIb3DQEBAQUABIIBAAiE/BrBKhGaGGDr6fuVcWAy+e0/CmVj67dC
# fKL0+0j7pU8+32Ct4YH5P6k11MnsreuAPScpmsPfCtWWbGoyf/LAQB5IwUfdtvSF
# SdGQ7UxhpNbohz1zn1DGtbZtpW8lyTWWjpMYRQqgV682n2B0cG5Bvc3AioQ+nBW4
# 0jkN9D2WTJzq/jilsc0vWWkvqOHUU0zPtSUQvjx5f0uAlFTQZg7Op+sfVh73Snt1
# Hp4bqBTmijPe4nOny8kowAYlCvjtF3D9e1GIBYqSZDL/kZwxiasKukFxG2y9AUBv
# ZnHPU5M1LAKLdET9oBKLQMYMYHavOaWfcXL3jie/cPlvcgbCjH2hghetMIIXqQYK
# KwYBBAGCNwMDATGCF5kwgheVBgkqhkiG9w0BBwKggheGMIIXggIBAzEPMA0GCWCG
# SAFlAwQCAQUAMIIBWgYLKoZIhvcNAQkQAQSgggFJBIIBRTCCAUECAQEGCisGAQQB
# hFkKAwEwMTANBglghkgBZQMEAgEFAAQg4/waBvKXFChQWA8aWtVlbXZBDTpzMkQJ
# K2dmmafcUacCBmbrdc3JxRgTMjAyNDA5MjcwNzUwNDMuOTQyWjAEgAIB9KCB2aSB
# 1jCB0zELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcT
# B1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEtMCsGA1UE
# CxMkTWljcm9zb2Z0IElyZWxhbmQgT3BlcmF0aW9ucyBMaW1pdGVkMScwJQYDVQQL
# Ex5uU2hpZWxkIFRTUyBFU046NjUxQS0wNUUwLUQ5NDcxJTAjBgNVBAMTHE1pY3Jv
# c29mdCBUaW1lLVN0YW1wIFNlcnZpY2WgghH7MIIHKDCCBRCgAwIBAgITMwAAAfWZ
# CZS88cZQjAABAAAB9TANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFt
# cCBQQ0EgMjAxMDAeFw0yNDA3MjUxODMxMDFaFw0yNTEwMjIxODMxMDFaMIHTMQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNy
# b3NvZnQgSXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxJzAlBgNVBAsTHm5TaGll
# bGQgVFNTIEVTTjo2NTFBLTA1RTAtRDk0NzElMCMGA1UEAxMcTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgU2VydmljZTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIB
# AMzvdHBUE1nf1j/OCE+yTCFtX0C+tbHX4JoZX09J72BG9pL5DRdO92cI73rklqLd
# /Oy4xNEwohvd3uiNB8yBUAZ28Rj/1jwVIqxau1hOUQLLoTX2FC/jyG/YyatwsFsS
# An8Obf6U8iDh4yr6NZUDk1mcqYq/6bGcBBO8trlgD22SUxaynp+Ue98dh28cuHlt
# Q3Jl48ptsBVr9dLAR+NGoyX3vjpMHE3aGK2NypKTf0UEo3snCtG4Y6NAhmCGGvmT
# AGqNEjUf0dSdWOrC5IgiTt2kK20tUs+5fv6iYMvH8hGTDQ+TLOwtLBGjr6AR4lkq
# UzOL3NMQywpnOjxr9NwrVrtiosqqy/AQAdRGMjkoSNyE+/WqwyA6y/nXvdRX45km
# wWOY/h70tJd3V5Iz9x6J/G++JVsIpBdK8xKxdJ95IVQLrMe0ptaBhvtOoc/VwMt1
# qLvk+knuqGuSw4kID031kf4/RTZPCbtOqEn04enNN1dLpZWtCMMvh81JflpmMRS1
# ND4ml7JoLnTcFap+dc6/gYt1zyfOcFrsuhhk+5wQ5lzc0zZMyvfAwUI0zmm0F1Gf
# POGG/QxTXIoJnlU2JMlF2eobHHfDcquOQNw925Pp157KICtWe82Y+l2xn7e8YDmL
# 73lOqdPn67YWxezF7/ouanA/R3xZjquFWB3r1XrGG+j9AgMBAAGjggFJMIIBRTAd
# BgNVHQ4EFgQUVeB8W/VKNKBw8CWSXttosXtgdQEwHwYDVR0jBBgwFoAUn6cVXQBe
# Yl2D9OXSZacbUzUZ6XIwXwYDVR0fBFgwVjBUoFKgUIZOaHR0cDovL3d3dy5taWNy
# b3NvZnQuY29tL3BraW9wcy9jcmwvTWljcm9zb2Z0JTIwVGltZS1TdGFtcCUyMFBD
# QSUyMDIwMTAoMSkuY3JsMGwGCCsGAQUFBwEBBGAwXjBcBggrBgEFBQcwAoZQaHR0
# cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNyb3NvZnQlMjBU
# aW1lLVN0YW1wJTIwUENBJTIwMjAxMCgxKS5jcnQwDAYDVR0TAQH/BAIwADAWBgNV
# HSUBAf8EDDAKBggrBgEFBQcDCDAOBgNVHQ8BAf8EBAMCB4AwDQYJKoZIhvcNAQEL
# BQADggIBAHMMZlT2gPcR337qJtEzkqdobKbn9RtHB1vylxwLoZ6VvP0r5auY/Wii
# P/PxunxiEDK9M5aWrvI8vNyOM3JRnSY5eUtNksQ5VCmsLVr4H+4nWtOj4I3kDNXl
# +C7reG2z309BRKe+xu+oYcrF8UyTR7+cvn8E4VHoonJYoWcKnGTKWuOpvqFeooE1
# OiNBJ53qLTKhbNEN8x4FVa+Fl45xtgXJ5IqeNnncoP/Yl3M6kwaxJL089FJZbaRR
# mkJy86vjaPFRIKtFBu1tRC2RoZpsRZhwAcE0+rDyRVevA3y6AtIgfUG2/VWfJr20
# 1eSbSEgZJU7lQJRJM14vSyIzZsfpJ3QXyj/HcRv8W0V6bUA0A2grEuqIC5MC4B+s
# 0rPrpfVpsyNBfMyJm4Z2YVM4iB4XhaOB/maKIz2HIEyuv925Emzmm5kBX/eQfAen
# uVql20ubPTnTHVJVtYEyNa+bvlgMB9ihu3cZ3qE23/42Jd01LT+wB6cnJNnNJ7p/
# 0NAsnKWvUFB/w8tNZOrUKJjVxo4r4NvwRnIGSdB8PAuilXpRCd9cS6BNtZvfjRIE
# igkaBRNS5Jmt9UsiGsp23WBG/LDpWcpzHZvMj5XQ8LheeLyYhAK463AzV3ugaG2V
# Ik1kir79QyWnUdUlAjvzndtRoFPoWarvnSoIygGHXkyL4vUdq7S2MIIHcTCCBVmg
# AwIBAgITMwAAABXF52ueAptJmQAAAAAAFTANBgkqhkiG9w0BAQsFADCBiDELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9z
# b2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMjEwOTMwMTgy
# MjI1WhcNMzAwOTMwMTgzMjI1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2Fz
# aGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
# cnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAx
# MDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAOThpkzntHIhC3miy9ck
# eb0O1YLT/e6cBwfSqWxOdcjKNVf2AX9sSuDivbk+F2Az/1xPx2b3lVNxWuJ+Slr+
# uDZnhUYjDLWNE893MsAQGOhgfWpSg0S3po5GawcU88V29YZQ3MFEyHFcUTE3oAo4
# bo3t1w/YJlN8OWECesSq/XJprx2rrPY2vjUmZNqYO7oaezOtgFt+jBAcnVL+tuhi
# JdxqD89d9P6OU8/W7IVWTe/dvI2k45GPsjksUZzpcGkNyjYtcI4xyDUoveO0hyTD
# 4MmPfrVUj9z6BVWYbWg7mka97aSueik3rMvrg0XnRm7KMtXAhjBcTyziYrLNueKN
# iOSWrAFKu75xqRdbZ2De+JKRHh09/SDPc31BmkZ1zcRfNN0Sidb9pSB9fvzZnkXf
# tnIv231fgLrbqn427DZM9ituqBJR6L8FA6PRc6ZNN3SUHDSCD/AQ8rdHGO2n6Jl8
# P0zbr17C89XYcz1DTsEzOUyOArxCaC4Q6oRRRuLRvWoYWmEBc8pnol7XKHYC4jMY
# ctenIPDC+hIK12NvDMk2ZItboKaDIV1fMHSRlJTYuVD5C4lh8zYGNRiER9vcG9H9
# stQcxWv2XFJRXRLbJbqvUAV6bMURHXLvjflSxIUXk8A8FdsaN8cIFRg/eKtFtvUe
# h17aj54WcmnGrnu3tz5q4i6tAgMBAAGjggHdMIIB2TASBgkrBgEEAYI3FQEEBQID
# AQABMCMGCSsGAQQBgjcVAgQWBBQqp1L+ZMSavoKRPEY1Kc8Q/y8E7jAdBgNVHQ4E
# FgQUn6cVXQBeYl2D9OXSZacbUzUZ6XIwXAYDVR0gBFUwUzBRBgwrBgEEAYI3TIN9
# AQEwQTA/BggrBgEFBQcCARYzaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9w
# cy9Eb2NzL1JlcG9zaXRvcnkuaHRtMBMGA1UdJQQMMAoGCCsGAQUFBwMIMBkGCSsG
# AQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1UdDwQEAwIBhjAPBgNVHRMBAf8EBTAD
# AQH/MB8GA1UdIwQYMBaAFNX2VsuP6KJcYmjRPZSQW9fOmhjEMFYGA1UdHwRPME0w
# S6BJoEeGRWh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3Rz
# L01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNybDBaBggrBgEFBQcBAQROMEwwSgYI
# KwYBBQUHMAKGPmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2VydHMvTWlj
# Um9vQ2VyQXV0XzIwMTAtMDYtMjMuY3J0MA0GCSqGSIb3DQEBCwUAA4ICAQCdVX38
# Kq3hLB9nATEkW+Geckv8qW/qXBS2Pk5HZHixBpOXPTEztTnXwnE2P9pkbHzQdTlt
# uw8x5MKP+2zRoZQYIu7pZmc6U03dmLq2HnjYNi6cqYJWAAOwBb6J6Gngugnue99q
# b74py27YP0h1AdkY3m2CDPVtI1TkeFN1JFe53Z/zjj3G82jfZfakVqr3lbYoVSfQ
# JL1AoL8ZthISEV09J+BAljis9/kpicO8F7BUhUKz/AyeixmJ5/ALaoHCgRlCGVJ1
# ijbCHcNhcy4sa3tuPywJeBTpkbKpW99Jo3QMvOyRgNI95ko+ZjtPu4b6MhrZlvSP
# 9pEB9s7GdP32THJvEKt1MMU0sHrYUP4KWN1APMdUbZ1jdEgssU5HLcEUBHG/ZPkk
# vnNtyo4JvbMBV0lUZNlz138eW0QBjloZkWsNn6Qo3GcZKCS6OEuabvshVGtqRRFH
# qfG3rsjoiV5PndLQTHa1V1QJsWkBRH58oWFsc/4Ku+xBZj1p/cvBQUl+fpO+y/g7
# 5LcVv7TOPqUxUYS8vwLBgqJ7Fx0ViY1w/ue10CgaiQuPNtq6TPmb/wrpNPgkNWcr
# 4A245oyZ1uEi6vAnQj0llOZ0dFtq0Z4+7X6gMTN9vMvpe784cETRkPHIqzqKOghi
# f9lwY1NNje6CbaUFEMFxBmoQtB1VM1izoXBm8qGCA1YwggI+AgEBMIIBAaGB2aSB
# 1jCB0zELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcT
# B1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEtMCsGA1UE
# CxMkTWljcm9zb2Z0IElyZWxhbmQgT3BlcmF0aW9ucyBMaW1pdGVkMScwJQYDVQQL
# Ex5uU2hpZWxkIFRTUyBFU046NjUxQS0wNUUwLUQ5NDcxJTAjBgNVBAMTHE1pY3Jv
# c29mdCBUaW1lLVN0YW1wIFNlcnZpY2WiIwoBATAHBgUrDgMCGgMVACbACruPDW0e
# WEYN1kgUAso83ZL2oIGDMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
# c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
# b3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIw
# MTAwDQYJKoZIhvcNAQELBQACBQDqoH/kMCIYDzIwMjQwOTI3MDA1MDQ0WhgPMjAy
# NDA5MjgwMDUwNDRaMHQwOgYKKwYBBAGEWQoEATEsMCowCgIFAOqgf+QCAQAwBwIB
# AAICC5IwBwIBAAICE70wCgIFAOqh0WQCAQAwNgYKKwYBBAGEWQoEAjEoMCYwDAYK
# KwYBBAGEWQoDAqAKMAgCAQACAwehIKEKMAgCAQACAwGGoDANBgkqhkiG9w0BAQsF
# AAOCAQEAMJWtfdo4Rr+5hUb2nKuop5TLV44AMhidU/CuP9thOinWdyHyXRuNVgii
# f4zqPDQvHz1RG509oDBoGFyffYxuleYYJizRbjfbHg8dLi6KVsHZ8VbI5DXBOFi9
# 2fXhRDr0sD64WXIPsV5zt6OaqD86qrAG+J96giMkUQdlPujioX+/yAFAgfTGBbks
# sw25UY0TCsa9YJY8QT7C8Ty4mhfswrtZU/Wymx3HFUntPIU0AmrEPsPBzObgFWRL
# MaE2cF5j6Mnuo59sviYAn90Vt5sJphng8d7fxhfh5mzCaZUn7uhXZ1ctdd4vNoSS
# DtnH7ei30xOiIryWxB+tt0ItOZdWLDGCBA0wggQJAgEBMIGTMHwxCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBU
# aW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAB9ZkJlLzxxlCMAAEAAAH1MA0GCWCGSAFl
# AwQCAQUAoIIBSjAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwLwYJKoZIhvcN
# AQkEMSIEILTT//8yGbSZYaAgSRxQkx/2mOBMXQ4RZcKnPqnw2WiKMIH6BgsqhkiG
# 9w0BCRACLzGB6jCB5zCB5DCBvQQgwdby0hcIdPSEruJHRL+P7YPXkdWJPMOce4+R
# k4amjzUwgZgwgYCkfjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMAITMwAA
# AfWZCZS88cZQjAABAAAB9TAiBCAy5WDeT3HIXrTp4YPV2l71K2Fc/NZclmIyd8LT
# REMMujANBgkqhkiG9w0BAQsFAASCAgBze6+Meoc0KW+6VHWYpRZzVpFuXquUf/iS
# BNQtqT7FbUDqIvTE44UpwsM4JqSH0/yqV/UGW4OUGmndmrAg3nE/ET0MXUnx9sQt
# D0q+IgHYy+EMmC2u6YhjHZ/fhNLpsQqa7u+zbd0JOFmKT8HY4M7GH5+Ee9Miyfci
# vD86jRIAEP7/84INNKnpvX+fSbUpWSnjFl6sW6Wgtfms/ytI5zaOEJTz3vR405Au
# jfURO0k3gb2DeBv4KnY3ALCULyx1eUkNHzar68J+fIxqL48Wlok5natgZfMHcI+e
# jCu1uuuLlljgjHIwDAPUA+oKzPcyXJi7TJxeNkpwWDpqRkhqNq0yyqPhdYZYXShC
# VZQzp0TDzn3/NRGp2OMhKkhaeBDsDycxNE4Vg0npw6PrOOpsFNXzHiYHWZChRwXt
# ol5PYolW7/hV5T1PQ62Ggy624AsRarnKu77ioSpyPvJmAZJSwByM1MlzhXZu0wNZ
# L6kt4fx3bwNQV1YPC86bQW7GEsZgrFugcIxQcQXlBSL2ZPh2cR64PMhltQWWdOQ4
# xhXmS+VVuCkjdwevKXXfC5UZyQBOK33ROq4qCrKMhpHQW1uAYTaHAEsz5PmIsRdi
# s3cXMkxgIfcFw5fxbLmL/uQ7WiT1j/lbS6OeieArAawpjzIGAjvopYbKsi4s2YSO
# bW0GRtHmIg==
# SIG # End signature block
